// Configuration
const API_ENDPOINT = 'api/get_call_logs.php';
const REFRESH_INTERVAL = 120000; // 2 minutes

// State
let autoRefreshEnabled = true;
let refreshInterval = null;

// DOM Elements
const callLogsContent = document.getElementById('callLogsContent');
const lastUpdatedElement = document.getElementById('lastUpdated');
const autoRefreshToggle = document.getElementById('autoRefreshToggle');
const refreshBtn = document.getElementById('refreshBtn');
const leaderboardContainer = document.getElementById('leaderboardContainer');

// Initialize the dashboard
document.addEventListener('DOMContentLoaded', () => {
    // Load initial data
    loadCallLogs();
    
    // Set up auto-refresh
    startAutoRefresh();
    
    // Set up event listeners
    if (autoRefreshToggle) {
        autoRefreshToggle.addEventListener('click', toggleAutoRefresh);
    }
    
    if (refreshBtn) {
        refreshBtn.addEventListener('click', loadCallLogs);
    }
});

// Load call logs from the API
async function loadCallLogs() {
    try {
        showLoading();
        const response = await fetch(API_ENDPOINT);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const responseData = await response.json();
        
        if (responseData.error) {
            showError(responseData.error);
            return;
        }
        
        renderLeaderboard(responseData.data.leaderboard);
        renderCallLogs(responseData.data.callLogs);
        updateStats(responseData.data.callLogs);
        updateLastUpdated();
    } catch (error) {
        console.error('Error loading call logs:', error);
        showError(`Failed to load call logs: ${error.message}`);
    }
}

// Render call logs to the page
function renderCallLogs(data) {
    if (!data || data.length === 0) {
        callLogsContent.innerHTML = '<div class="no-data">No call logs found for today.</div>';
        return;
    }

    const html = data.map(user => {
        const totalSeconds = timeStringToSeconds(user.total_time);
        const goalSeconds = 3600; // 1 hour
        const progress = Math.min((totalSeconds / goalSeconds) * 100, 100);
        const goalMet = totalSeconds >= goalSeconds;

        return `
            <div class="user-card">
                <div class="user-header">
                    <div class="user-name">${escapeHtml(user.name)}</div>
                    <div class="last-call">Last call: ${user.last_call_ago || 'N/A'}</div>
                </div>
                <div class="call-stats">
                    <div class="stat-item outbound">
                        <div class="stat-item-value">${user.outbound_count || 0}</div>
                        <div class="stat-item-label">Outbound Calls</div>
                    </div>
                    <div class="stat-item outbound">
                        <div class="stat-item-value">${user.outbound_time || '00:00:00'}</div>
                        <div class="stat-item-label">Outbound Time</div>
                    </div>
                    <div class="stat-item inbound">
                        <div class="stat-item-value">${user.inbound_count || 0}</div>
                        <div class="stat-item-label">Inbound Calls</div>
                    </div>
                    <div class="stat-item inbound">
                        <div class="stat-item-value">${user.inbound_time || '00:00:00'}</div>
                        <div class="stat-item-label">Inbound Time</div>
                    </div>
                    <div class="stat-item total">
                        <div class="stat-item-value">${user.total_time || '00:00:00'}</div>
                        <div class="stat-item-label">Total Time</div>
                    </div>
                </div>
                <div class="goal-progress-container">
                    <div class="goal-progress-bar ${goalMet ? 'goal-met' : ''}" style="width: ${progress}%;"></div>
                </div>
            </div>
        `;
    }).join('');
    
    callLogsContent.innerHTML = html;
}

// Helper function to convert HH:MM:SS to seconds
function timeStringToSeconds(timeStr) {
    if (!timeStr) return 0;
    const parts = timeStr.split(':').map(Number);
    if (parts.length !== 3) return 0;
    return parts[0] * 3600 + parts[1] * 60 + parts[2];
}

// Render the leaderboard
function renderLeaderboard(leaderboardData) {
    if (!leaderboardData || leaderboardData.length === 0) {
        leaderboardContainer.innerHTML = ''; // Clear if no data
        return;
    }

    const html = `
        <div class="leaderboard-title">Weekly Top Agents (Total Time)</div>
        <div class="leaderboard">
            ${leaderboardData.map((user, index) => `
                <div class="leaderboard-item rank-${index + 1}">
                    <div class="rank">#${index + 1}</div>
                    <div class="name">${escapeHtml(user.name)}</div>
                    <div class="time">${user.total_time}</div>
                </div>
            `).join('')}
        </div>
    `;
    
    leaderboardContainer.innerHTML = html;
}

// Update the stats cards
function updateStats(data) {
    if (!data || data.length === 0) return;
    
    // Calculate totals
    const totals = data.reduce((acc, item) => {
        acc.totalCalls += (item.outbound_count || 0) + (item.inbound_count || 0);
        acc.totalTime += timeToSeconds(item.total_time || '00:00:00');
        return acc;
    }, { totalCalls: 0, totalTime: 0 });
    
    // Update DOM
    document.getElementById('totalCalls').textContent = totals.totalCalls;
    document.getElementById('totalTime').textContent = secondsToTime(totals.totalTime);
    document.getElementById('activeUsers').textContent = data.length;
    
    const avgCallTime = totals.totalCalls > 0 
        ? Math.round(totals.totalTime / totals.totalCalls) 
        : 0;
    document.getElementById('avgCallTime').textContent = secondsToTime(avgCallTime);
}

// Toggle auto-refresh
function toggleAutoRefresh() {
    autoRefreshEnabled = !autoRefreshEnabled;
    
    if (autoRefreshEnabled) {
        startAutoRefresh();
        if (autoRefreshToggle) {
            autoRefreshToggle.classList.add('active');
        }
    } else {
        stopAutoRefresh();
        if (autoRefreshToggle) {
            autoRefreshToggle.classList.remove('active');
        }
    }
}

// Start auto-refresh
function startAutoRefresh() {
    if (refreshInterval) {
        clearInterval(refreshInterval);
    }
    refreshInterval = setInterval(loadCallLogs, REFRESH_INTERVAL);
}

// Stop auto-refresh
function stopAutoRefresh() {
    if (refreshInterval) {
        clearInterval(refreshInterval);
        refreshInterval = null;
    }
}

// Show loading state
function showLoading() {
    callLogsContent.innerHTML = '<div class="loading">Loading call logs...</div>';
}

// Show error message
function showError(message) {
    callLogsContent.innerHTML = `
        <div class="error-message">
            <p>${escapeHtml(message)}</p>
            <button onclick="loadCallLogs()" class="btn">Retry</button>
        </div>
    `;
}

// Update the last updated timestamp
function updateLastUpdated() {
    const now = new Date();
    lastUpdatedElement.textContent = `Last updated: ${now.toLocaleTimeString()}`;
}

// Helper: Convert time string to seconds
function timeToSeconds(timeStr) {
    if (!timeStr) return 0;
    
    const parts = timeStr.split(':');
    if (parts.length !== 3) return 0;
    
    return (parseInt(parts[0]) * 3600) + (parseInt(parts[1]) * 60) + parseInt(parts[2]);
}

// Helper: Convert seconds to HH:MM:SS
function secondsToTime(seconds) {
    const hrs = Math.floor(seconds / 3600);
    const mins = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    return [
        hrs.toString().padStart(2, '0'),
        mins.toString().padStart(2, '0'),
        secs.toString().padStart(2, '0')
    ].join(':');
}

// Helper: Escape HTML to prevent XSS
function escapeHtml(unsafe) {
    if (typeof unsafe !== 'string') return '';
    return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}
