<?php

namespace Controllers;

use Exception;
use getID3;

require_once 'app/getid3/getid3.php';

class Playlist

{
    // Properties
    public array $playlist = array();

    // function to check passed in filename exists and is readable return true or false
    public function checkFile($fileName): bool
    {
        if (!is_readable($fileName)) {
            return false;
        }
        return true;
    }
    // function to check passed in folder path exists and is readable return true or false
    public function checkFolder($path): bool
    {
        if (!is_dir($path)) {
            return false;
        }
        return true;
    }
    // function using getid3 to extract playtime from mp4 file
    public function getVideoDuration($fileName) : int
    {
        $getID3 = new getID3;
        $file = $getID3->analyze($fileName);
        return intval(round($file['playtime_seconds']));
    }

    // function to create playlist from Google Slides playlist

    /**
     * @throws Exception
     */
    public function createNewsPlaylist($fileName): void
    {
        // check file exists/is readable
        if (!is_readable($fileName)) {
            throw new Exception("The file $fileName does not exist");
        }
        // get count for number of items (lines) in array
        $lines = count(file($fileName));
        // check if file is empty and return and empty session playlist array

        if($lines < 1) {
            $_SESSION['playlist'] = array();
            return;
        }
        // create playlist from file
        $this->createFromFile($fileName);
        // set session playlist variable
        $_SESSION['playlist'] = $this->playlist;
    }

    // function to create slide content playlist from csv file
    public function createFromFile($fileName): array
    {
        // check file exists/is readable
        $this->checkFile($fileName);

        // read in csv file and create an array from it
        $rows   = array_map('str_getcsv', file($fileName));

        // create header from first row
        $header = array_shift($rows);

        // create the holding array
        $csvArr    = array();
        // loop through rows and combine each with header to create key value pairs
        foreach($rows as $row) {
            $csvArr[] = array_combine($header, $row);
        }

        // get count for number of items (lines) in array
        $rowCount = count($csvArr);

        // loop through slide array and find all that could be playing today
        for ($i = 0; $i < $rowCount; $i++) {
            $dateOn = $csvArr[$i]['date_on'];
            $dateOff = $csvArr[$i]['date_off'];
            $currentDate = date("Y-m-d");
            if ($dateOn <= $currentDate && $dateOff >= $currentDate) {
                // if slides should be displayed add to playlist
                $item = [
                    'id' => $csvArr[$i]['id'],
                    'description' => $csvArr[$i]['description'],
                    'time_on' => $csvArr[$i]['time_on'],
                    'date_on' => $csvArr[$i]['date_on'],
                    'time_off' => $csvArr[$i]['time_off'],
                    'date_off' => $csvArr[$i]['date_off'],
                    'link' => $csvArr[$i]['link'],
                    'play_for' => $csvArr[$i]['duration'],
                    'display_on_idle' => $csvArr[$i]['display_on_idle'],
                    'type' => 'link'
                ];
                $this->playlist[] = $item;
            }
        }

        return $this->playlist;
    }
    // create array of mp4 files from folder to add to playlist - $path is the folder path
    // used as an alternative to csv file
    public function createFromFolder($path): array
    {
        $files = glob($path . '*.mp4');

        foreach($files as $file) {
            // get duration of video
            $duration = $this->getVideoDuration($file);
            $item = [
                'id' => NULL,
                'description' => 'N/A',
                'time_on' => NULL,
                'date_on' => NULL,
                'time_off' => NULL,
                'date_off' => NULL,
                'link' => $file,
                'play_for' => $duration,
                'display_on_idle' => 1,
                'type' => 'mp4'
            ];
            $this->playlist[] = $item;
        }
        return $this->playlist;
    }
    // create playlist of mp4 files from csv file
    public function createFromCSV($fileName): void
    {

        // check file exists/is readable
        if (!is_readable($fileName)) {
            echo "The file $fileName does not exist";
            exit;
        }

        // read in csv file and create an array from it
        $rows   = array_map('str_getcsv', file($fileName));

        // create header from first row
        $header = array_shift($rows);

        // create the holding array
        $csvArr    = array();
        // loop through rows and combine each with header to create key value pairs
        foreach($rows as $row) {
            $csvArr[] = array_combine($header, $row);
        }
        // get count for number of items (lines) in array
        $rowCount = count($csvArr);

        // loop through item array and add to playlist
        for ($i = 0; $i < $rowCount; $i++) {
            /* check if date_on and date_off are set and if not set to NULL * - not currently used */
            $dateOn = NULL;
            $dateOff = NULL;
            $currentDate = date("Y-m-d");

            $item = [
                'id' => NULL,
                'description' => NULL,
                'time_on' => NULL,
                'date_on' => NULL,
                'time_off' => NULL,
                'date_off' => NULL,
                'link' => $csvArr[$i]['link'],
                'play_for' => $csvArr[$i]['play_for'],
                'display_on_idle' => $csvArr[$i]['display_on_idle'],
                'type' => 'mp4'
            ];
            $this->playlist[] = $item;
            }
}
    // create files.csv from mp4 files in specified folder
    public function createFilesCSV($path): bool|array|string
    {
        // check folder exists
        if (!$this->checkFolder($path)) {
            return "The folder $path does not exist";
        }

        // find all mp4 files in folder
        $files = glob($path . '*.mp4');

        // create csv file
        $csv = fopen($path . 'files.csv', 'w');

        // add header
        fputcsv($csv, ['id', 'description', 'time_on', 'date_on', 'time_off', 'date_off', 'link', 'play_for', 'display_on_idle', 'type']);

        // Initialize an array to store errors
        $errors = [];

        // loop through files and add to csv
        foreach ($files as $file) {
            // get duration of video
            $duration = $this->getVideoDuration($file);

            // If duration is zero, add to errors and skip this file
            if ($duration === 0) {
                $errors[] = "The file $file has a duration of zero.";
                continue;
            }

            $item = [
                'id' => NULL,
                'description' => 'N/A',
                'time_on' => NULL,
                'date_on' => NULL,
                'time_off' => NULL,
                'date_off' => NULL,
                'link' => $file,
                'play_for' => $duration,
                'display_on_idle' => 1,
                'type' => 'mp4'
            ];

            // add item to csv
            fputcsv($csv, $item);
        }

        // close csv file
        fclose($csv);

        // Return errors if there are any, otherwise return true
        return !empty($errors) ? $errors : true;
    }

    // validates a CSV file line by line and returns an array of errors
    public function validateCSV($path): array|string|null
    {
        // check file exists/is readable
        if (!$this->checkFile($path)) {
            return "The file $path does not exist";
        }

        // Open the CSV file
        $file = fopen($path, 'r');

        // Get the first line and store it as the header
        $header = fgetcsv($file);

        // Initialize an empty array to store the errors
        $errors = array();

        // Initialize a variable to store the line number
        $lineNumber = 1;

        // Loop through the rest of the lines in the CSV file
        while (($line = fgetcsv($file)) !== FALSE) {
            $lineNumber++;
            // Check if the number of elements in the line matches the number of elements in the header
            if (count($header) !== count($line)) {
                // If they don't match, add an error message to the errors array
                $errors[] = "Line $lineNumber: Number of elements does not match the header";
            }
        }

        // Close the CSV file
        fclose($file);

        // Return the errors array if there are errors, otherwise return null
        return count($errors) > 0 ? $errors : null;
    }

    // returns the playlist array from $this->playlist
    public function getPlaylistAsArray(): array
    {
        return $this->playlist;
    }


}