from flask import Flask, jsonify, render_template_string, request
from flask_limiter import Limiter
from flask_limiter.util import get_remote_address
import os
import time
from ringcentral import SDK
import json
from datetime import datetime, timezone
from functools import wraps
import logging

# Configure logging for better debugging
logging.basicConfig(level=logging.DEBUG, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

app = Flask(__name__)

# Security configurations
app.config['SECRET_KEY'] = os.environ.get('SECRET_KEY', 'change-this-secret-key-in-production')

# Rate limiting to prevent abuse
limiter = Limiter(
    key_func=get_remote_address,
    default_limits=["100 per hour", "10 per minute"]
)
limiter.init_app(app)

# RingCentral configuration - move to environment variables in production
config = {
    "client_id": "1CayFASqF2Qdv1MFI2nDgo",
    "client_secret": "Xim1wYJpsORcp3d4KrwrjHZr9t2ckc8H9cvPkhtmHxpV",
    "server_url": "https://platform.ringcentral.com",
    "jwt": {
        "Reports": "eyJraWQiOiI4NzYyZjU5OGQwNTk0NGRiODZiZjVjYTk3ODA0NzYwOCIsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJodHRwczovL3BsYXRmb3JtLnJpbmdjZW50cmFsLmNvbS9yZXN0YXBpL29hdXRoL3Rva2VuIiwic3ViIjoiMjIzNzIxMDQyIiwiaXNzIjoiaHR0cHM6Ly9wbGF0Zm9ybS5yaW5nY2VudHJhbC5jb20iLCJleHAiOjIyNDAyNjU1OTksImlhdCI6MTcwMTEwMzI5MywianRpIjoiYkxGZDJqOTVSUWk2THROcW1JLVhFZyJ9.ALtTujcoO5CNQmkh-YX-tMibPE2Szx3Kzt8PsEKPetygnW23HCpKtaXufDjgG_qMe5PxfcuGQgLITU3gGh_H4k4kSIlviyJTrWwiwwOcBTsorNbIQMjbVwNdpMZGfzCf1bqkAFq7FIJMfR0KopXSBSL5RsnkWa12Vm_uUEQBIpiZd0T0F9uVJeaVcQlOPr4dK1zae3XEtqubdWT6xyAqli-glTj4IDgn-VNfM2WGk63QMb6ApraALmwulNCjqNPK6bJ7Q2U65zMPRDJmfy-jVpdS7KABBuj76bt0itBwt_-EdYp92DjB8-aHEsJiaLOYMW2bKuAqq95hHlJbnQ45fQ"
    }
}

hidden_users = ["Emma Crane", "Nathan Oliver", "Fran Ferreria", "Elaine Fogarty"]

def format_duration(seconds):
    """Convert seconds to HH:MM:SS format"""
    if seconds < 0:
        seconds = 0
    
    hours = seconds // 3600
    minutes = (seconds % 3600) // 60
    secs = seconds % 60
    
    return f"{hours:02d}:{minutes:02d}:{secs:02d}"

def get_best_duration(record):
    """Extract the best duration from a call record"""
    # Handle milliseconds vs seconds properly
    duration_ms = record.get('durationMs', 0)
    duration_seconds = record.get('duration', 0)
    
    # Convert milliseconds to seconds if available
    if duration_ms and duration_ms > 0:
        calculated_duration = int(duration_ms / 1000)  # Convert ms to seconds
    elif duration_seconds and duration_seconds > 0:
        calculated_duration = int(duration_seconds)    # Already in seconds
    else:
        calculated_duration = 0
    
    # Try other duration fields as backup
    backup_duration_fields = [
        ('billableDuration', record.get('billableDuration', 0)),
        ('talkTime', record.get('talkTime', 0))
    ]
    
    # Calculate legs duration
    legs_duration = 0
    legs = record.get('legs', [])
    if legs:
        for leg in legs:
            # Check if leg has durationMs or duration
            leg_duration_ms = leg.get('durationMs', 0)
            leg_duration_s = leg.get('duration', 0)
            
            if leg_duration_ms and leg_duration_ms > 0:
                legs_duration += int(leg_duration_ms / 1000)
            elif leg_duration_s and leg_duration_s > 0:
                legs_duration += int(leg_duration_s)
    
    # Log all duration options for debugging
    user_name = record.get('from', {}).get('name', 'Unknown')
    logger.debug(f"Duration options for {user_name}:")
    logger.debug(f"  durationMs: {duration_ms}ms -> {int(duration_ms/1000) if duration_ms else 0}s")
    logger.debug(f"  duration: {duration_seconds}s")
    logger.debug(f"  calculated_duration: {calculated_duration}s ({format_duration(calculated_duration)})")
    
    for field_name, value in backup_duration_fields:
        logger.debug(f"  {field_name}: {value}s ({format_duration(value)})")
    
    logger.debug(f"  legs_total: {legs_duration}s ({format_duration(legs_duration)})")
    
    # Choose the best duration (prefer calculated, then backup fields, then legs)
    best_duration = calculated_duration
    
    if best_duration == 0:
        # Try backup fields
        for field_name, value in backup_duration_fields:
            if isinstance(value, (int, float)) and value > 0:
                best_duration = max(best_duration, int(value))
    
    if best_duration == 0:
        # Use legs duration as last resort
        best_duration = legs_duration
    
    logger.debug(f"  Selected duration: {best_duration}s ({format_duration(best_duration)})")
    return best_duration

def fetch_call_logs():
    """Fetch call logs from RingCentral API"""
    try:
        logger.info("Starting to fetch call logs...")
        
        rcsdk = SDK(config["client_id"], config["client_secret"], config["server_url"])
        platform = rcsdk.platform()
        
        # Login with JWT
        logger.info("Authenticating with RingCentral...")
        platform.login(jwt=config["jwt"]["Reports"])
        
        # Get today's date in ISO format
        today_iso = datetime.utcnow().strftime('%Y-%m-%dT00:00:00.000Z')
        logger.info(f"Fetching calls from: {today_iso}")
        
        # Retry logic for rate limiting
        max_retries = 5
        retry_delay = 1
        
        # First fetch outbound calls
        outbound_records = []
        for attempt in range(max_retries):
            try:
                logger.info(f"Outbound API call attempt {attempt + 1}/{max_retries}")
                response = platform.get('/restapi/v1.0/account/~/call-log', {
                    'dateFrom': today_iso,
                    'type': 'Voice',
                    'direction': 'Outbound',
                    'perPage': 1000
                })
                outbound_records = json.loads(response.text()).get('records', [])
                logger.info(f"Retrieved {len(outbound_records)} outbound records")
                break
            except Exception as e:
                logger.error(f"Outbound API call failed: {str(e)}")
                if 'Request rate exceeded' in str(e):
                    logger.info(f"Rate limited, waiting {retry_delay} seconds...")
                    time.sleep(retry_delay)
                    retry_delay *= 2
                else:
                    raise e
        
        # Reset retry delay for next API call
        retry_delay = 1
        
        # Then fetch inbound calls
        inbound_records = []
        for attempt in range(max_retries):
            try:
                logger.info(f"Inbound API call attempt {attempt + 1}/{max_retries}")
                response = platform.get('/restapi/v1.0/account/~/call-log', {
                    'dateFrom': today_iso,
                    'type': 'Voice',
                    'direction': 'Inbound',
                    'perPage': 1000
                })
                inbound_records = json.loads(response.text()).get('records', [])
                logger.info(f"Retrieved {len(inbound_records)} inbound records")
                if inbound_records:
                    logger.debug("Sample inbound record:")
                    logger.debug(json.dumps(inbound_records[0], indent=2))
                break
            except Exception as e:
                logger.error(f"Inbound API call failed: {str(e)}")
                if 'Request rate exceeded' in str(e):
                    logger.info(f"Rate limited, waiting {retry_delay} seconds...")
                    time.sleep(retry_delay)
                    retry_delay *= 2
                else:
                    raise e
        
        # Combine all records
        all_records = outbound_records + inbound_records
        logger.info(f"Total records to process: {len(all_records)}")
        
        # Debug: Log structure of first record if available
        if all_records:
            sample = all_records[0]
            logger.debug("Sample record structure:")
            logger.debug(json.dumps(sample, indent=2, default=str))
            logger.debug(f"Sample direction: {sample.get('direction')}")
            
        user_data = {}
        processed_records = 0
        
        for record in all_records:
            direction = record.get('direction')
            if direction not in ['Inbound', 'Outbound']:
                continue
                
            # For outbound, use from.name; for inbound, use to[0].name
            if direction == 'Outbound':
                user_info = record.get('from', {})
            else:  # Inbound
                to_field = record.get('to', [])
                if isinstance(to_field, list):
                    user_info = to_field[0] if to_field else {}
                else:
                    user_info = to_field # It's already a dictionary
                
            user_name = user_info.get('name')
            if direction == 'Inbound':
                logger.debug(f"Inbound call processed for user: {user_name}")
            
            # Skip if no name or in hidden users
            if not user_name or user_name in hidden_users:
                continue
            
            # Get call duration
            duration = get_best_duration(record)
            
            # Get start time
            start_time_str = record.get('startTime')
            if not start_time_str:
                logger.warning(f"No start time for record: {record.get('id', 'Unknown')}")
                continue
                
            try:
                start_time = datetime.fromisoformat(start_time_str.replace("Z", "+00:00"))
            except ValueError as e:
                logger.error(f"Invalid start time format: {start_time_str}, error: {e}")
                continue
            
            # Initialize user data if not exists
            if user_name not in user_data:
                user_data[user_name] = {
                    "outbound": {"total_duration": 0, "call_count": 0, "last_call_time": None},
                    "inbound": {"total_duration": 0, "call_count": 0, "last_call_time": None},
                    "last_call_time": None
                }
            
            # Update direction-specific data
            direction_key = direction.lower()
            user_data[user_name][direction_key]["total_duration"] += duration
            user_data[user_name][direction_key]["call_count"] += 1
            
            # Update last call time for this direction
            if (user_data[user_name][direction_key]["last_call_time"] is None or 
                start_time > user_data[user_name][direction_key]["last_call_time"]):
                user_data[user_name][direction_key]["last_call_time"] = start_time
            
            # Update overall last call time
            if (user_data[user_name]["last_call_time"] is None or 
                start_time > user_data[user_name]["last_call_time"]):
                user_data[user_name]["last_call_time"] = start_time
            
            processed_records += 1
        
        logger.info(f"Processed {processed_records} call records")
        logger.info(f"Found {len(user_data)} unique users")
        
        # Format results
        results = []
        now = datetime.now(timezone.utc)
        
        # Log user totals for debugging
        logger.info("User call totals:")
        for user, data in sorted(user_data.items(), key=lambda x: -(x[1]["outbound"]["total_duration"] + x[1]["inbound"]["total_duration"])):
            outbound_duration = data["outbound"]["total_duration"]
            outbound_count = data["outbound"]["call_count"]
            inbound_duration = data["inbound"]["total_duration"]
            inbound_count = data["inbound"]["call_count"]
            total_duration = outbound_duration + inbound_duration
            
            logger.info(f"  {user}:")
            logger.info(f"    Outbound: {outbound_count} calls, {outbound_duration}s total ({format_duration(outbound_duration)})")
            logger.info(f"    Inbound: {inbound_count} calls, {inbound_duration}s total ({format_duration(inbound_duration)})")
            
            # Calculate time since last call
            if data["last_call_time"] is None:
                last_call_str = "no calls"
            else:
                time_diff = now - data["last_call_time"]
                minutes_ago = int(time_diff.total_seconds() // 60)
                
                if minutes_ago < 1:
                    last_call_str = "just now"
                elif minutes_ago < 60:
                    last_call_str = f"{minutes_ago} minute{'s' if minutes_ago != 1 else ''} ago"
                else:
                    hours_ago = minutes_ago // 60
                    remaining_minutes = minutes_ago % 60
                    if remaining_minutes == 0:
                        last_call_str = f"{hours_ago} hour{'s' if hours_ago != 1 else ''} ago"
                    else:
                        last_call_str = f"{hours_ago}h {remaining_minutes}m ago"
            
            results.append({
                "name": user,
                "outbound_time": format_duration(outbound_duration),
                "outbound_count": outbound_count,
                "inbound_time": format_duration(inbound_duration),
                "inbound_count": inbound_count,
                "total_time": format_duration(total_duration),
                "last_call_ago": last_call_str
            })
        
        logger.info(f"Returning {len(results)} user records")
        return results
        
    except Exception as e:
        logger.error(f"Error fetching call logs: {str(e)}", exc_info=True)
        # In case of an error, return an empty list to avoid breaking the frontend
        return []

@app.route('/')
def dashboard():
    """Serve the dashboard HTML"""
    return render_template_string(r"""
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Call Logs Dashboard</title>
        <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600;700&display=swap" rel="stylesheet">
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }

            body {
                font-family: 'Poppins', sans-serif;
                background: #f4f7fc;
                color: #333;
                min-height: 100vh;
                padding: 20px;
            }

            .container {
                max-width: 1200px;
                margin: 0 auto;
            }

            .header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 40px;
                color: #002366; /* TMG Blue */
            }
            
            .header-left {
                display: flex;
                align-items: center;
            }

            .logo {
                height: 50px;
                margin-right: 20px;
            }

            .header h1 {
                font-size: 2rem;
                font-weight: 600;
            }

            .header-right p {
                font-size: 1rem;
                opacity: 0.9;
                text-align: right;
            }

            .auto-refresh-info {
                font-size: 0.8rem;
                opacity: 0.7;
                margin-top: 5px;
            }

            .stats-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin-bottom: 40px;
            }

            .stat-card {
                background: #ffffff;
                border-radius: 15px;
                padding: 25px;
                text-align: center;
                box-shadow: 0 8px 32px rgba(0,0,0,0.05);
                border: 1px solid #e0e0e0;
                transition: transform 0.3s ease, box-shadow 0.3s ease;
            }

            .stat-card:hover {
                transform: translateY(-5px);
                box-shadow: 0 12px 40px rgba(0, 35, 102, 0.1);
            }

            .stat-value {
                font-size: 2rem;
                font-weight: 700;
                color: #002366; /* TMG Blue */
                margin-bottom: 5px;
            }

            .stat-label {
                font-size: 0.9rem;
                color: #555;
                text-transform: uppercase;
            }

            #users-container {
                margin-top: 20px;
            }

            .user-card {
                background: #ffffff;
                border-radius: 15px;
                padding: 20px;
                margin-bottom: 20px;
                box-shadow: 0 4px 20px rgba(0,0,0,0.05);
                border: 1px solid #e0e0e0;
                transition: all 0.3s ease;
            }

            .user-card:hover {
                transform: translateY(-3px);
                box-shadow: 0 10px 30px rgba(0, 35, 102, 0.1);
            }

            .user-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 15px;
                padding-bottom: 10px;
                border-bottom: 1px solid #eee;
            }

            .user-name {
                font-size: 1.3rem;
                font-weight: 600;
                color: #002366; /* TMG Blue */
            }

            .last-call {
                font-size: 0.9rem;
                color: #7f8c8d;
            }

            .call-stats {
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 15px;
                margin-top: 15px;
            }

            .stat-item {
                text-align: center;
                padding: 12px;
                border-radius: 10px;
                background: #f8f9fa;
                transition: all 0.3s ease;
                border: 1px solid #e0e0e0;
            }

            .stat-item:hover {
                background: #f1f3f5;
                transform: translateY(-2px);
            }

            .stat-item.outbound {
                border-left: 4px solid #007bff;
            }

            .stat-item.inbound {
                border-left: 4px solid #28a745;
            }

            .stat-item.total {
                border-left: 4px solid #6f42c1;
                background: #f8f5ff;
            }

            .stat-item-label {
                font-size: 0.8rem;
                color: #64748b;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                margin-top: 5px;
            }

            .stat-item-value {
                font-size: 1.2rem;
                font-weight: 700;
                color: #1e293b;
            }

            .call-type {
                display: inline-block;
                padding: 3px 8px;
                border-radius: 12px;
                font-size: 0.7rem;
                font-weight: 600;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }

            .call-type.outbound {
                background: #e0e7ff;
                color: #007bff;
            }

            .call-type.inbound {
                background: #d4edda;
                color: #28a745;
            }

            .loading, .no-data {
                text-align: center;
                padding: 30px;
                color: #555;
                font-size: 1.2rem;
            }

            .error-message {
                background: #f8d7da;
                color: #721c24;
                padding: 15px;
                border-radius: 8px;
                margin: 20px 0;
                text-align: center;
                display: none;
            }

            @media (max-width: 768px) {
                .header {
                    flex-direction: column;
                    align-items: flex-start;
                }
                .call-stats {
                    grid-template-columns: 1fr 1fr;
                }
                
                .stat-item {
                    padding: 10px;
                }
                
                .stat-item-value {
                    font-size: 1.1rem;
                }
            }

            @media (max-width: 480px) {
                .call-stats {
                    grid-template-columns: 1fr;
                }
                
                .header h1 {
                    font-size: 1.8rem;
                }
            }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <div class="header-left">
                    <img src="https://i.imgur.com/oOpwKqY.png" alt="Time Media Group Logo" class="logo">
                    <h1>Call Logs Dashboard</h1>
                </div>
                <div class="header-right">
                    <p>Today's Call Activity</p>
                    <div class="auto-refresh-info">Auto-refreshes every 2 minutes</div>
                </div>
            </div>

            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-value" id="totalCalls">0</div>
                    <div class="stat-label">Total Calls</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="totalTime">00:00:00</div>
                    <div class="stat-label">Total Talk Time</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="activeUsers">0</div>
                    <div class="stat-label">Active Users</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="avgCallTime">00:00:00</div>
                    <div class="stat-label">Avg Call Time</div>
                </div>
            </div>

            <div class="call-logs-container">
                <div class="controls">
                    <h2 class="section-title">Call Activity by User</h2>
                    <div class="refresh-controls">
                        <div class="auto-refresh-toggle">
                            <span>Auto-refresh</span>
                            <div class="toggle-switch active" onclick="toggleAutoRefresh()" id="autoRefreshToggle">
                                <div class="toggle-slider"></div>
                            </div>
                        </div>
                        <button class="refresh-btn" onclick="loadCallLogs()" id="refreshBtn">🔄 Refresh Now</button>
                    </div>
                </div>
                <div id="callLogsContent">
                    <div class="loading">Loading call logs...</div>
                </div>
                <div class="last-updated" id="lastUpdated"></div>
            </div>
        </div>

        <script>
            let autoRefreshInterval = null;
            let autoRefreshEnabled = true;

            function parseTimeToSeconds(timeStr) {
                // Parse HH:MM:SS format
                const parts = timeStr.split(':');
                if (parts.length === 3) {
                    return parseInt(parts[0]) * 3600 + parseInt(parts[1]) * 60 + parseInt(parts[2]);
                }
                return 0;
            }

            function formatSecondsToTime(totalSeconds) {
                const hours = Math.floor(totalSeconds / 3600);
                const minutes = Math.floor((totalSeconds % 3600) / 60);
                const seconds = totalSeconds % 60;
                
                return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }

            function updateStats(data) {
                const totalCalls = data.reduce((sum, item) => sum + (item.outbound_count || 0) + (item.inbound_count || 0), 0);
                const totalSeconds = data.reduce((sum, item) => sum + parseTimeToSeconds(item.outbound_time) + parseTimeToSeconds(item.inbound_time), 0);
                const avgSeconds = totalCalls > 0 ? Math.round(totalSeconds / totalCalls) : 0;
                const activeUsers = data.length;

                document.getElementById('totalCalls').textContent = totalCalls;
                document.getElementById('totalTime').textContent = formatSecondsToTime(totalSeconds);
                document.getElementById('activeUsers').textContent = activeUsers;
                document.getElementById('avgCallTime').textContent = formatSecondsToTime(avgSeconds);
            }

            function renderCallLogs(data) {
                const container = document.getElementById('callLogsContent');
                
                if (!data || data.length === 0) {
                    container.innerHTML = '<div class="no-data">No call logs found for today.</div>';
                    return;
                }

                const html = data.map(item => `
                    <div class="call-log-item">
                        <div class="user-info">
                            <div class="user-name">${item.name}</div>
                        </div>
                        <div class="call-stats">
                            <div class="stat-item">
                                <div class="stat-item-value">${item.outbound_count || 0}</div>
                                <div class="stat-item-label">Outbound Calls</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.outbound_time}</div>
                                <div class="stat-item-label">Outbound Time</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.inbound_count || 0}</div>
                                <div class="stat-item-label">Inbound Calls</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.inbound_time}</div>
                                <div class="stat-item-label">Inbound Time</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.total_time}</div>
                                <div class="stat-item-label">Total Time</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.last_call_ago}</div>
                                <div class="stat-item-label">Last Call</div>
                            </div>
                        </div>
                    </div>
                `).join('');

                container.innerHTML = html;
            }

            async function loadCallLogs() {
                const container = document.getElementById('callLogsContent');
                const refreshBtn = document.getElementById('refreshBtn');
                const lastUpdated = document.getElementById('lastUpdated');
                
                container.innerHTML = '<div class="loading">Loading call logs...</div>';
                refreshBtn.disabled = true;
                refreshBtn.textContent = '🔄 Loading...';

                try {
                    const response = await fetch('/api/call-logs');
                    
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    
                    const data = await response.json();
                    renderCallLogs(data);
                    updateStats(data);
                    
                    const now = new Date();
                    lastUpdated.textContent = `Last updated: ${now.toLocaleTimeString()}`;
                    
                } catch (error) {
                    console.error('Error loading call logs:', error);
                    container.innerHTML = `<div class="error">Failed to load call logs: ${error.message}</div>`;
                } finally {
                    refreshBtn.disabled = false;
                    refreshBtn.textContent = '🔄 Refresh Now';
                }
            }

            function toggleAutoRefresh() {
                const toggle = document.getElementById('autoRefreshToggle');
                autoRefreshEnabled = !autoRefreshEnabled;
                
                if (autoRefreshEnabled) {
                    toggle.classList.add('active');
                    startAutoRefresh();
                } else {
                    toggle.classList.remove('active');
                    stopAutoRefresh();
                }
            }

            function startAutoRefresh() {
                if (autoRefreshInterval) {
                    clearInterval(autoRefreshInterval);
                }
                
                autoRefreshInterval = setInterval(() => {
                    if (autoRefreshEnabled) {
                        loadCallLogs();
                    }
                }, 120000); // 2 minutes
            }

            function stopAutoRefresh() {
                if (autoRefreshInterval) {
                    clearInterval(autoRefreshInterval);
                    autoRefreshInterval = null;
                }
            }

            // Load data when page loads and start auto-refresh
            document.addEventListener('DOMContentLoaded', () => {
                loadCallLogs();
                startAutoRefresh();
            });

            // Clean up on page unload
            window.addEventListener('beforeunload', stopAutoRefresh);
        </script>
    </body>
    </html>
    """)

@app.route('/api/call-logs', methods=['GET'])
@limiter.limit("5 per minute")
def get_call_logs():
    """API endpoint to fetch call logs"""
    try:
        call_logs = fetch_call_logs()
        return jsonify(call_logs)
    except Exception as e:
        logger.error(f"API error: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/health', methods=['GET'])
def health_check():
    """Health check endpoint"""
    return jsonify({'status': 'healthy', 'timestamp': datetime.utcnow().isoformat()}), 200

# Security headers middleware
@app.after_request
def after_request(response):
    response.headers['X-Content-Type-Options'] = 'nosniff'
    response.headers['X-Frame-Options'] = 'DENY'
    response.headers['X-XSS-Protection'] = '1; mode=block'
    response.headers['Referrer-Policy'] = 'strict-origin-when-cross-origin'
    if request.is_secure:
        response.headers['Strict-Transport-Security'] = 'max-age=31536000; includeSubDomains'
    return response

if __name__ == '__main__':
    # For development only - use proper WSGI server in production
    app.run(
        host='127.0.0.1',
        port=5000,
        debug=True
    )