// Office Chair Race — fun, cartoony dashboard
const API_URL = '../api/get_call_logs.php'; // same host, relative to /calllogs/race/
const REFRESH_MS = 60000; // 60s
let lastLeader = null;
let autoTimer = null;

const track = document.getElementById('track');
const finish = document.getElementById('finish');
const legendList = document.getElementById('legendList');
const leaderToast = document.getElementById('leaderToast');
const leaderName = document.getElementById('leaderName');
const lastUpdated = document.getElementById('lastUpdated');

const metricSel = document.getElementById('metric');
const autoupdateSel = document.getElementById('autoupdate');
const refreshBtn = document.getElementById('refreshBtn');

const confettiCanvas = document.getElementById('confettiCanvas');
const ctx = confettiCanvas.getContext('2d');
let confettiPieces = [];

function resizeCanvas(){
  confettiCanvas.width = window.innerWidth;
  confettiCanvas.height = window.innerHeight;
}
window.addEventListener('resize', resizeCanvas); resizeCanvas();

function secondsFromHHMMSS(s){
  if(!s) return 0;
  const parts = s.split(':');
  if(parts.length !== 3) return 0;
  const [h,m,sec] = parts.map(Number);
  return h*3600 + m*60 + sec;
}
function statFor(item, metric){
  if(metric==='count'){
    return (item.inbound_count||0) + (item.outbound_count||0);
  }else{
    return secondsFromHHMMSS(item.inbound_time) + secondsFromHHMMSS(item.outbound_time);
  }
}

function colorForName(name){
  let h = 0;
  for (let i=0;i<name.length;i++){ h = (h*31 + name.charCodeAt(i)) % 360; }
  return `hsl(${h} 80% 60%)`;
}
function avatarFor(name){
  const key = name.toLowerCase();
  if(key.includes('jonathan brown')) return 'chairs/jonathan_brown.svg';
  if(key.includes('yusuf')) return 'chairs/yusuf_awad.svg';
  if(key.includes('joe hall')) return 'chairs/joe_hall.svg';
  if(key.includes('andrew parkinson')) return 'chairs/andrew_parkinson.svg';
  if(key.includes('jon valentine')) return 'chairs/jon_valentine.svg';
  if(key.includes('emma crane')) return 'chairs/emma_crane.svg';
  if(key.includes('nathan oliver')) return 'chairs/nathan_oliver.svg';
  return 'chairs/default.svg';
}

function createLane(i, total){
  const lane = document.createElement('div');
  lane.className = 'lane';
  lane.style.top = (i*(track.clientHeight/Math.max(total,1))) + 'px';
  lane.style.height = (track.clientHeight/Math.max(total,1)) + 'px';
  track.appendChild(lane);
  return lane;
}

function createRacer(item, idx){
  const wrap = document.createElement('div');
  wrap.className = 'racer';
  wrap.dataset.name = item.name;

  const badge = document.createElement('div');
  badge.className = 'badge';
  const av = document.createElement('img');
  av.src = avatarFor(item.name);
  av.className = 'avatar';

  const nameEl = document.createElement('div');
  nameEl.className = 'name';
  const emoji = '💼';
  nameEl.textContent = `${item.name} ${emoji}`;
  const stat = document.createElement('div');
  stat.className = 'stat';
  stat.textContent = `📞 ${(item.inbound_count||0)+(item.outbound_count||0)} • ⏱️ ${item.inbound_time}+${item.outbound_time}`;

  badge.appendChild(av);
  const txtBox = document.createElement('div');
  txtBox.appendChild(nameEl);
  txtBox.appendChild(stat);
  badge.appendChild(txtBox);

  const chair = document.createElement('img');
  chair.src = 'chairs/chair.svg';
  chair.className = 'chair';

  wrap.appendChild(chair);
  wrap.appendChild(badge);
  track.appendChild(wrap);
  return wrap;
}

function updateLegend(data, metric){
  legendList.innerHTML='';
  const sorted = [...data].sort((a,b)=> statFor(b,metric)-statFor(a,metric));
  sorted.forEach((it, i)=>{
    const row = document.createElement('div');
    row.className = 'row';
    const dot = document.createElement('div');
    dot.className = 'dot';
    dot.style.background = colorForName(it.name);
    const label = document.createElement('div');
    const s = statFor(it, metric);
    const statText = metric==='count' ? `${s} calls` : `${Math.round(s/60)} min`;
    label.innerHTML = `<b>${i+1}.</b> ${it.name} — ${statText}`;
    row.appendChild(dot);
    row.appendChild(label);
    legendList.appendChild(row);
  });
}

function layoutRace(data, metric){
  track.querySelectorAll('.lane,.racer').forEach(e=>e.remove());

  const total = data.length;
  const finishX = track.clientWidth - 160;
  finish.style.right = '24px';

  const values = data.map(d => statFor(d, metric));
  const max = Math.max(1, ...values);

  data.forEach((item, idx)=>{
    const lane = createLane(idx, total);
    const racer = createRacer(item, idx);
    const v = statFor(item, metric);

    const x = Math.max(10, Math.floor((v / max) * (finishX - 40)));
    const y = lane.offsetTop + lane.clientHeight/2 - 40;
    racer.style.transform = `translate(${x}px, ${y}px)`;
  });
}

function animateToPositions(data, metric){
  const finishX = track.clientWidth - 160;
  const values = data.map(d => statFor(d, metric));
  const max = Math.max(1, ...values);
  data.forEach(item=>{
    const racer = [...document.querySelectorAll('.racer')].find(r=>r.dataset.name===item.name);
    if(!racer) return;
    const v = statFor(item, metric);
    const x = Math.max(10, Math.floor((v / max) * (finishX - 40)));
    const current = racer.style.transform;
    racer.classList.add('moving');
    racer.style.transform = current.replace(/translate\([^)]*\)/, (m)=>{
      const y = m.match(/,\s*([\d.]+)px\)/);
      const ypx = y ? y[1] : 0;
      return `translate(${x}px, ${ypx}px)`;
    });
    setTimeout(()=>racer.classList.remove('moving'), 900);
  });
}

function toastLeader(name){
  leaderName.textContent = name;
  leaderToast.classList.remove('hidden');
  setTimeout(()=> leaderToast.classList.add('hidden'), 2500);
}

function makeConfetti(){
  for(let i=0;i<160;i++){
    confettiPieces.push({
      x: Math.random()*confettiCanvas.width,
      y: -20 - Math.random()*200,
      s: 5+Math.random()*6,
      vy: 2 + Math.random()*3,
      vx: -1+Math.random()*2,
      rot: Math.random()*Math.PI,
      vr: -0.1 + Math.random()*0.2,
      color: `hsl(${Math.floor(Math.random()*360)} 90% 60%)`
    });
  }
}
function drawConfetti(){
  ctx.clearRect(0,0,confettiCanvas.width, confettiCanvas.height);
  confettiPieces.forEach(p=>{
    ctx.save();
    ctx.translate(p.x,p.y);
    ctx.rotate(p.rot);
    ctx.fillStyle=p.color;
    ctx.fillRect(-p.s/2,-p.s/2,p.s,p.s);
    ctx.restore();
    p.x += p.vx; p.y += p.vy; p.rot += p.vr;
  });
  confettiPieces = confettiPieces.filter(p=> p.y < confettiCanvas.height + 50);
  requestAnimationFrame(drawConfetti);
}

function detectLeaderChange(data, metric){
  const sorted = [...data].sort((a,b)=> statFor(b,metric) - statFor(a,metric));
  const leader = sorted[0]?.name || null;
  if(leader && leader !== lastLeader){
    if(lastLeader !== null){
      toastLeader(leader);
      makeConfetti();
    }
    lastLeader = leader;
  }
}

function cacheSet(key, value){
  try { localStorage.setItem(key, JSON.stringify(value)); } catch(e){}
}
function cacheGet(key){
  try { return JSON.parse(localStorage.getItem(key)||'null'); } catch(e){ return null; }
}

async function fetchData(){
  const url = API_URL;
  const r = await fetch(url, {cache:'no-store'});
  if(!r.ok) throw new Error('HTTP '+r.status);
  const j = await r.json();
  if(j.status !== 'success') throw new Error(j.message || 'API error');
  let arr = Array.isArray(j.data) ? j.data : (j.data?.callLogs || []);
  return arr;
}

async function refresh(){
  const metric = metricSel.value;
  try{
    const data = await fetchData();
    if(!track.querySelector('.racer')){
      layoutRace(data, metric);
    }else{
      animateToPositions(data, metric);
    }
    updateLegend(data, metric);
    detectLeaderChange(data, metric);
    cacheSet('race_last_data', {t: Date.now(), metric, data});
    lastUpdated.textContent = 'Last updated: ' + new Date().toLocaleTimeString();
  }catch(err){
    console.error('Refresh failed', err);
    const cached = cacheGet('race_last_data');
    if(cached){
      if(!track.querySelector('.racer')){
        layoutRace(cached.data, cached.metric);
      }
      updateLegend(cached.data, cached.metric);
    }
  }
}

function startAuto(){
  if(autoupdateSel.value === 'on'){
    if(autoTimer) clearInterval(autoTimer);
    autoTimer = setInterval(refresh, REFRESH_MS);
  }else{
    if(autoTimer) clearInterval(autoTimer);
    autoTimer = null;
  }
}

refreshBtn.addEventListener('click', refresh);
metricSel.addEventListener('change', refresh);
autoupdateSel.addEventListener('change', startAuto);

drawConfetti();
refresh();
startAuto();
