/* ===========================================================
   Call Logs Dashboard Script
   Compatible with the new get_call_logs.php JSON format
   =========================================================== */

async function loadCallLogs() {
  try {
    const response = await fetch("api/get_call_logs.php", { cache: "no-store" });
    const result = await response.json();

    if (!result || result.status !== "success") {
      console.error("API error:", result ? result.message : "No response");
      showError("Failed to load call data");
      return;
    }

    const logs = Array.isArray(result.data) ? result.data : [];

    if (logs.length === 0) {
      document.querySelector("#no-data").style.display = "block";
      clearTable();
      updateSummaryStats([]);
      return;
    }

    document.querySelector("#no-data").style.display = "none";
    renderCallLogs(logs);
    updateSummaryStats(logs);

  } catch (err) {
    console.error("Error loading call logs:", err);
    showError("Error loading call logs: " + err.message);
  }
}

/* -----------------------------
   Render table rows
--------------------------------*/
function renderCallLogs(logs) {
  const tbody = document.querySelector("#call-table");
  if (!tbody) return;

  tbody.innerHTML = "";

  logs.forEach((log) => {
    const row = document.createElement("tr");
    row.innerHTML = `
      <td>${sanitize(log.name)}</td>
      <td>${log.inbound_count}</td>
      <td>${log.outbound_count}</td>
      <td>${log.total_time}</td>
    `;
    tbody.appendChild(row);
  });
}

/* -----------------------------
   Update summary statistics
--------------------------------*/
function updateSummaryStats(logs) {
  const totalCalls = logs.reduce(
    (sum, x) => sum + x.inbound_count + x.outbound_count,
    0
  );
  const totalTimeSec = logs.reduce(
    (sum, x) => sum + timeToSeconds(x.total_time),
    0
  );
  const avgCallSec = totalCalls > 0 ? totalTimeSec / totalCalls : 0;

  const activeUsers = logs.length;

  setText("#total-calls", totalCalls);
  setText("#total-time", formatSeconds(totalTimeSec));
  setText("#avg-time", formatSeconds(avgCallSec));
  setText("#active-users", activeUsers);
}

/* -----------------------------
   Utility helpers
--------------------------------*/
function timeToSeconds(hms) {
  if (!hms) return 0;
  const [h, m, s] = hms.split(":").map(Number);
  return h * 3600 + m * 60 + s;
}

function formatSeconds(sec) {
  const h = String(Math.floor(sec / 3600)).padStart(2, "0");
  const m = String(Math.floor((sec % 3600) / 60)).padStart(2, "0");
  const s = String(Math.floor(sec % 60)).padStart(2, "0");
  return `${h}:${m}:${s}`;
}

function sanitize(str) {
  return (str || "")
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;");
}

function setText(selector, val) {
  const el = document.querySelector(selector);
  if (el) el.textContent = val;
}

function clearTable() {
  const tbody = document.querySelector("#call-table");
  if (tbody) tbody.innerHTML = "";
}

function showError(msg) {
  console.warn(msg);
  const el = document.querySelector("#error-message");
  if (el) {
    el.textContent = msg;
    el.style.display = "block";
  }
}

/* -----------------------------
   Auto-refresh every 2 minutes
--------------------------------*/
document.addEventListener("DOMContentLoaded", () => {
  loadCallLogs();
  setInterval(loadCallLogs, 120000); // 2 min
});
