<?php

$dbPath = __DIR__ . '/../sponsor_content.db';

try {
    // Connect to the database
    $db = new PDO('sqlite:' . $dbPath);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Start transaction
    $db->beginTransaction();
    
    // First, check if the table exists and has data
    $tableExists = $db->query("SELECT name FROM sqlite_master WHERE type='table' AND name='sponsors'")->fetchColumn();
    
    if (!$tableExists) {
        die("Sponsors table does not exist. Please run previous migrations first.\n");
    }
    
    // Get the current table structure
    $oldTableInfo = $db->query("PRAGMA table_info(sponsors)")->fetchAll(PDO::FETCH_ASSOC);
    
    // Create a mapping of old columns to new columns
    $columnMapping = [
        'id' => 'id',
        'name' => 'business_name',
        'image_path' => 'button',
        'video_path' => 'mp4',
        'display_order' => 'display_order',
        'is_active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at'
    ];
    
    // Create the new table
    $db->exec("DROP TABLE IF EXISTS sponsors_new");
    
    $createTableSQL = "
    CREATE TABLE sponsors_new (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        business_name TEXT,
        contract TEXT DEFAULT '',
        address_1 TEXT DEFAULT '',
        address_2 TEXT DEFAULT '',
        street TEXT DEFAULT '',
        city TEXT DEFAULT '',
        county TEXT DEFAULT '',
        postal_code TEXT DEFAULT '',
        country TEXT DEFAULT '',
        phone TEXT DEFAULT '',
        mobile TEXT DEFAULT '',
        email TEXT DEFAULT '',
        url TEXT DEFAULT '',
        social TEXT DEFAULT '',
        banner TEXT DEFAULT '',
        button TEXT,
        mp4 TEXT,
        sort_order INTEGER DEFAULT 0,
        is_active INTEGER DEFAULT 1,
        display_order INTEGER DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )";
    
    $db->exec($createTableSQL);
    
    // Build the column lists for the INSERT query
    $sourceColumns = [];
    $targetColumns = [];
    
    foreach ($oldTableInfo as $column) {
        $colName = $column['name'];
        if (isset($columnMapping[$colName])) {
            $targetCol = $columnMapping[$colName];
            $sourceColumns[] = $colName;
            $targetColumns[] = $targetCol;
        }
    }
    
    // Add new columns with default values
    $newColumns = [
        'contract', 'address_1', 'address_2', 'street', 'city', 'county',
        'postal_code', 'country', 'phone', 'mobile', 'email', 'url',
        'social', 'banner', 'sort_order'
    ];
    
    // Only include new columns that aren't already in the mapping
    foreach ($newColumns as $col) {
        if (!in_array($col, $targetColumns)) {
            $targetColumns[] = $col;
            $sourceColumns[] = "'' AS $col";
        }
    }
    
    $sourceColumnsList = implode(', ', $sourceColumns);
    $targetColumnsList = implode(', ', $targetColumns);
    
    // Copy data from old table to new table
    $copySQL = "
        INSERT INTO sponsors_new ($targetColumnsList)
        SELECT $sourceColumnsList FROM sponsors
    ";
    
    $db->exec($copySQL);
    
    // Drop the old table and rename the new one
    $db->exec("DROP TABLE IF EXISTS sponsors_old");
    $db->exec("ALTER TABLE sponsors RENAME TO sponsors_old");
    $db->exec("ALTER TABLE sponsors_new RENAME TO sponsors");
    
    // Commit the transaction
    $db->commit();
    
    echo "Sponsors table structure updated successfully.\n";
    
} catch (PDOException $e) {
    // Rollback the transaction in case of errors
    if (isset($db) && $db->inTransaction()) {
        $db->rollBack();
    }
    die("Database error: " . $e->getMessage() . "\n");
}
