<?php

$dbPath = __DIR__ . '/../sponsor_content.db';

try {
    $db = new PDO('sqlite:' . $dbPath);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Start transaction
    $db->beginTransaction();
    
    echo "Fixing sponsors table...\n";
    
    // Create a backup of the current table
    $db->exec("CREATE TABLE IF NOT EXISTS sponsors_backup AS SELECT * FROM sponsors");
    echo "- Created backup as 'sponsors_backup'\n";
    
    // Get current table info
    $columns = [];
    $result = $db->query("PRAGMA table_info(sponsors)");
    while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
        $columns[$row['name']] = $row['type'];
    }
    
    // Add missing columns if they don't exist
    $requiredColumns = [
        'id' => 'INTEGER PRIMARY KEY AUTOINCREMENT',
        'business_name' => 'TEXT',
        'contract' => 'TEXT DEFAULT ""',
        'address_1' => 'TEXT DEFAULT ""',
        'address_2' => 'TEXT DEFAULT ""',
        'street' => 'TEXT DEFAULT ""',
        'city' => 'TEXT DEFAULT ""',
        'county' => 'TEXT DEFAULT ""',
        'postal_code' => 'TEXT DEFAULT ""',
        'country' => 'TEXT DEFAULT ""',
        'phone' => 'TEXT DEFAULT ""',
        'mobile' => 'TEXT DEFAULT ""',
        'email' => 'TEXT DEFAULT ""',
        'url' => 'TEXT DEFAULT ""',
        'social' => 'TEXT DEFAULT ""',
        'banner' => 'TEXT DEFAULT ""',
        'button' => 'TEXT',
        'mp4' => 'TEXT',
        'sort_order' => 'INTEGER DEFAULT 0',
        'is_active' => 'INTEGER DEFAULT 1',
        'display_order' => 'INTEGER DEFAULT 0',
        'created_at' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP',
        'updated_at' => 'TIMESTAMP DEFAULT CURRENT_TIM'
    ];
    
    foreach ($requiredColumns as $column => $type) {
        if (!array_key_exists($column, $columns)) {
            $db->exec("ALTER TABLE sponsors ADD COLUMN $column $type");
            echo "- Added column: $column $type\n";
        }
    }
    
    // If name exists but business_name doesn't, copy data from name to business_name
    if (array_key_exists('name', $columns) && !array_key_exists('business_name', $columns)) {
        $db->exec("ALTER TABLE sponsors ADD COLUMN business_name TEXT");
        $db->exec("UPDATE sponsors SET business_name = name");
        echo "- Copied data from 'name' to 'business_name'\n";
    }
    
    // If image_path exists but button doesn't, copy data from image_path to button
    if (array_key_exists('image_path', $columns) && !array_key_exists('button', $columns)) {
        $db->exec("ALTER TABLE sponsors ADD COLUMN button TEXT");
        $db->exec("UPDATE sponsors SET button = image_path");
        echo "- Copied data from 'image_path' to 'button'\n";
    }
    
    // If video_path exists but mp4 doesn't, copy data from video_path to mp4
    if (array_key_exists('video_path', $columns) && !array_key_exists('mp4', $columns)) {
        $db->exec("ALTER TABLE sponsors ADD COLUMN mp4 TEXT");
        $db->exec("UPDATE sponsors SET mp4 = video_path");
        echo "- Copied data from 'video_path' to 'mp4'\n";
    }
    
    // Commit the changes
    $db->commit();
    
    echo "\nSponsors table has been fixed successfully!\n";
    
} catch (PDOException $e) {
    // Rollback the transaction in case of errors
    if (isset($db) && $db->inTransaction()) {
        $db->rollBack();
    }
    die("Error: " . $e->getMessage() . "\n");
}

// Run the check script to verify the changes
include __DIR__ . '/check_sponsor_content_db.php';
