<?php /** @noinspection PhpMissingFieldTypeInspection */

/** @noinspection PhpMissingFieldTypeInspection */

namespace Controllers;

use App\SQLiteConnection;
use Exception;
use PDO;
use PDOException;

class SponsorsController extends BaseController
{
    private $limit ;
    private $offset;

    protected \PDO $pdo;

    const DEFAULT_LIMIT = 15;

    public function __construct(PDO $pdo)
    {
        parent::__construct($pdo);
        $this->pdo = $pdo;
        $this->limit = (int)($_GET['limit'] ?? self::DEFAULT_LIMIT);
        $this->offset = (int)($_GET['offset'] ?? 0);
    }

    private function getTotalCount()
    {
        $stmt = $this->pdo->prepare("SELECT count(*) FROM sponsors");
        $stmt->execute();
        return $stmt->fetchColumn();
    }

    public function checkDatabaseConnection(): void
    {
        try {
            if ($this->pdo === null) {
                throw new Exception('Whoops, could not connect to the SQLite database!');
            }
        } catch (Exception $e) {
            echo $e->getMessage();
            exit();
        }
    }

    public function getSponsors($id = null)
    {
        if ($id === null) {
            $sql = "SELECT id, business_name, button, email FROM sponsors ORDER BY sort_order, business_name LIMIT $this->limit OFFSET $this->offset";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute();
            return $stmt->fetchAll();
        } else {
            //$id = (int)$id; // Ensure $id is an integer
            $sql = "SELECT * FROM sponsors WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->execute();
            $sponsor = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($sponsor === false) {
                error_log("No sponsor found with ID $id");
            } else {
                error_log("Sponsor found: " . print_r($sponsor, true));
            }

            return $sponsor;
        }
    }

    public function showAllSponsors(): void
    {
        $this->checkDatabaseConnection();
        $sponsors = $this->getSponsors();
        $sponsorsTotal = $this->getTotalCount();
        $sponsorCount = count($sponsors);
        $nextOffset = $this->limit + $this->offset;

        require 'public/views/show_all_sponsors.view.php';
    }



    public function showSingleSponsor($id): void
    {
        // Check if $id is an array and extract the value - addresses issue created by index.php when parameters are passed in the URL
        $id = $this->extractID($id);
        $this->checkDatabaseConnection();
        $sponsor = $this->getSponsors($id);
        // build concatenated address string
        if (!$sponsor['address_1'] == NULL) {
            $sponsorAddressDisplay = $sponsor['address_1'];
            if (!$sponsor['address_2'] == NULL)
                $sponsorAddressDisplay .= ", " . $sponsor['address_2'];
            if (!$sponsor['city'] == NULL)
                $sponsorAddressDisplay .= ", " . $sponsor['city'];
            if (!$sponsor['county'] == NULL)
                $sponsorAddressDisplay .= ", " . $sponsor['county'];
            if (!$sponsor['postal_code'] == NULL)
                $sponsorAddressDisplay .= " " . $sponsor['postal_code'];
        } else {
            $sponsorAddressDisplay = NULL;
        }
        if ($sponsor) {
            require 'public/views/show_single_sponsor.view.php';
        } else {
            echo "No sponsor found with ID $id";
        }
    }

    // show sponsor QR view page
    public function showSponsorQR($id): void
    {
        $this->checkDatabaseConnection();
        $id = $this->extractID($id);
        $sponsor = $this->getSponsors($id);
        $sponsorAddressDisplay = $this->formatSponsorAddress($sponsor);
        require 'public/views/show_sponsor_qr.view.php';
    }

    private function formatSponsorAddress(array $sponsor): ?string
    {
        $addressParts = array_filter([
            $sponsor['address_1'] ?? null,
            $sponsor['address_2'] ?? null,
            $sponsor['street'] ?? null,
            $sponsor['city'] ?? null,
            $sponsor['county'] ?? null,
            $sponsor['postal_code'] ?? null
        ]);

        return !empty($addressParts) ? implode(', ', $addressParts) : null;
    }



    public function editSponsor($id): void
    {
        $sql = "SELECT * FROM sponsors WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        $sponsor = $stmt->fetch(PDO::FETCH_ASSOC);

        require 'public/views/utilities/edit_sponsor.view.php';
    }

    public function updateSponsor($data): bool|string
    {
        $id = $data['id'];
        $requiredFields = ['id', 'contract', 'business_name', 'phone', 'button', 'mp4'];
        $missingFields = [];

        foreach ($requiredFields as $field) {
            if (empty($data[$field])) {
                $missingFields[] = $field;
            }
        }

        if (!empty($missingFields)) {
            return "Missing fields: " . implode(', ', $missingFields);
        }

        $sql = "UPDATE sponsors SET
                    contract = :contract,
                    business_name = :business_name,
                    address_1 = :address_1,
                    address_2 = :address_2,
                    street = :street,
                    city = :city,
                    county = :county,
                    postal_code = :postal_code,
                    phone = :phone,
                    mobile = :mobile,
                    email = :email,
                    url = :url,
                    social = :social,
                    button = :button,
                    mp4 = :mp4
                WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);

        try {
            $stmt->execute([
                ':contract' => $data['contract'],
                ':business_name' => $data['business_name'],
                ':address_1' => $data['address_1'],
                ':address_2' => $data['address_2'],
                ':street' => $data['street'],
                ':city' => $data['city'],
                ':county' => $data['county'],
                ':postal_code' => $data['postal_code'],
                ':phone' => $data['phone'],
                ':mobile' => $data['mobile'],
                ':email' => $data['email'],
                ':url' => $data['url'],
                ':social' => $data['social'],
                ':button' => $data['button'],
                ':mp4' => $data['mp4'],
                ':id' => $id
            ]);
            return true;
        } catch (PDOException $e) {
            return $e->getMessage();
        }
    }

    public function deleteSponsor($id): bool|string
    {
        $sql = "DELETE FROM sponsors WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);

        try {
            $stmt->execute([':id' => $id]);
            return true;
        } catch (PDOException $e) {
            return $e->getMessage();
        }
    }
}