<?php

namespace Controllers;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use InvalidArgumentException;
use JetBrains\PhpStorm\NoReturn;
use PDO;
use Utilities\ConfigSettings;

require 'vendor/autoload.php';

class RegisterSiteController
{
    private ConfigSettings $configSettings;
    private Client $client;

    public function __construct(PDO $pdo, ConfigSettings $configSettings = null)
    {
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        $this->pdo = $pdo;
        $this->configSettings = $configSettings ?? new ConfigSettings('public/config/site.cfg');
        $this->client = new Client(); // Instantiate Guzzle client here
    }

    public function getSiteSettings(): array
    {
        return $this->configSettings->getSettings();
    }

    public function run($type): void
    {
        if ($type === 'messages') {
            $this->registerForMessages();
        } elseif ($type === 'content') {
            $this->registerForContent();
        } else {
            throw new InvalidArgumentException("Invalid registration type specified.");
        }
    }

    #[NoReturn] public function registerForMessages(): void
    {
        $settings = $this->getSiteSettings();
        $messagingRegistrationRoute = $settings['url_to_messages_api'];

        try {
            $response = $this->client->post($messagingRegistrationRoute, [
                'form_params' => [
                    'company_id' => $settings['company_id'],
                    'site_ref' => $settings['site_ref'],
                    'site_location' => $settings['site_name'] . ', ' . $settings['site_location']
                ]
            ]);
        } catch (GuzzleException $e) {
            $_SESSION['error'] = 'Registration for messages unsuccessful';
            header('Location: /utilities');
            exit();
        }

        $data = json_decode($response->getBody(), true);

        if (!isset($data['id'])) {
            $_SESSION['error'] = 'Registration for messages unsuccessful';
        } else {
            $_SESSION['success'] = 'Registration for messages successful';
        }

        header('Location: /utilities');
        exit();
    }

    #[NoReturn] public function registerForContent(): void
    {
        $settings = $this->getSiteSettings();
        $contentRegistrationRoute = $settings['url_to_content_api'] . 'register-content';
        try {
            $response = $this->client->post($contentRegistrationRoute, [
                'form_params' => [
                    'site_group_id' => $settings['company_id'],
                    'site_ref' => $settings['site_ref'],
                    'site_name' => $settings['site_name'],
                    'site_address' => $settings['site_location'],
                    'site_postcode' => $settings['site_postcode'],
                    'site_country' => $settings['site_country'],
                    'site_email' => $settings['site_contact']
                ]
            ]);
        } catch (GuzzleException $e) {
            $_SESSION['error'] = 'Registration for content unsuccessful';
            header('Location: /utilities');
            exit();
        }

        $data = json_decode($response->getBody(), true);

        if (!isset($data['id'])) {
            $_SESSION['error'] = 'Registration for content unsuccessful';
        } else {
            $_SESSION['success'] = 'Registration for content successful';
        }

        header('Location: /utilities');
        exit();
    }
}