<?php
/**
 * Path Helper for Home Hardware PWA
 * 
 * This file provides helper functions for managing file and URL paths
 */

// Default configuration
$defaultConfig = [
    'basePath' => '/HomeHardware_PWA/HomeHardware/htdocs',
    'apiEndpoint' => '/HomeHardware_PWA/HomeHardware/htdocs/api',
    'assetsPath' => '/HomeHardware_PWA/HomeHardware/htdocs/public/app_assets',
    'iconsPath' => '/HomeHardware_PWA/HomeHardware/htdocs/icons',
    'viewsPath' => '/HomeHardware_PWA/HomeHardware/htdocs/public/views',
    'partialsPath' => '/HomeHardware_PWA/HomeHardware/htdocs/public/views/partials'
];

// Try to load configuration from file
$config = [];
$configFile = __DIR__ . '/config.json';

if (file_exists($configFile)) {
    $config = json_decode(file_get_contents($configFile), true) ?: [];
} else {
    // Create default config file if it doesn't exist
    file_put_contents($configFile, json_encode($defaultConfig, JSON_PRETTY_PRINT));
}

// Merge with defaults
$GLOBALS['config'] = array_merge($defaultConfig, $config);

// Only declare functions if they don't already exist
if (!function_exists('base_url')) {
    /**
     * Get the base URL
     * 
     * @param string $path The path to append to the base URL
     * @return string The full URL
     */
    function base_url($path = '') {
        static $baseUrl = null;
        
        if ($baseUrl === null) {
            $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
            $baseUrl = $protocol . $_SERVER['HTTP_HOST'] . $GLOBALS['config']['basePath'];
        }
        
        return rtrim($baseUrl, '/') . '/' . ltrim($path, '/');
    }
}

if (!function_exists('asset_url')) {
    /**
     * Get the path to the assets directory
     * 
     * @param string $path The path to append to the assets directory
     * @return string The full path to the asset
     */
    function asset_url($path = '') {
        return base_url('public/app_assets/' . ltrim($path, '/'));
    }
}

if (!function_exists('icon_url')) {
    /**
     * Get the path to the icons directory
     * 
     * @param string $path The path to append to the icons directory
     * @return string The full path to the icon
     */
    function icon_url($path = '') {
        return base_url('icons/' . ltrim($path, '/'));
    }
}

if (!function_exists('view_path')) {
    /**
     * Get the path to a view file
     * 
     * @param string $view The view name (without .php extension)
     * @return string The full path to the view file
     */
    function view_path($view) {
        return __DIR__ . '/public/views/' . ltrim($view, '/') . '.php';
    }
}

if (!function_exists('view')) {
    /**
     * Include a view file with variables
     * 
     * @param string $view The view name (without .php extension)
     * @param array $data The data to extract and make available in the view
     * @return void
     */
    function view($view, $data = []) {
        // Extract the data array into variables
        extract($data);
        
        // Start output buffering
        ob_start();
        
        // Include the view file
        include view_path($view);
        
        // Get the contents of the buffer and clean it
        return ob_get_clean();
    }
}

if (!function_exists('current_path')) {
    /**
     * Get the current URL path
     * 
     * @return string The current URL path
     */
    function current_path() {
        return parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    }
}

if (!function_exists('is_current_path')) {
    /**
     * Check if the current path matches the given pattern
     * 
     * @param string $pattern The pattern to match against
     * @return bool True if the current path matches the pattern, false otherwise
     */
    function is_current_path($pattern) {
        $current = current_path();
        return preg_match('~^' . preg_quote($pattern, '~') . '(/|\?|$|#|$)~', $current);
    }
}

// Add the base URL to the config for JavaScript
$GLOBALS['config']['baseUrl'] = base_url();
